/* $Header: /home/vikas/src/nocol/genmon/RCS/genmon.c,v 1.7 1998/07/15 16:27:32 vikas Exp $ */

/* Copyright 1994 Vikas Aggarwal, vikas@navya.com */
/* Copyright 1993 JvNCnet */

/*+ 
 * FUNCTION:
 * 	A generic monitor for nocol, to allow people to develop
 * monitors using shell scripts, etc.
 *
 * Does the initial opening of the datafile,
 * etc. and then does a popen() to the  program specified on the
 * command line. Reads back data line by line from the program and
 * stuffs it into the NOCOL structure and writes out the structure to
 * the datafile.
 *
 * Without any command line arguments, it acts like a filter, reading
 * NOCOL text lines from stdin and writing out the structure to stdout.
 *
 * Recognizes two additional keywords-  LOG and APPEND.  'LOG' means
 * that this program should do an eventlog()  and APPEND means to add
 * at the end of the existing DATAFILE on open(). Typically, the program
 * overwrites the datafile each time.
 *
 * AUTHOR:
 *
 *	Vikas Aggarwal, vikas@navya.com
 *
 */

/*+ 
 * $Log: genmon.c,v $
 * Revision 1.7  1998/07/15 16:27:32  vikas
 * Changed 'daemon()' to daemonize() since some C libs have daemon()
 *
 * Revision 1.6  1997/12/16 18:01:52  vikas
 * Changed the goto statement to a while() loop instead.
 *
 * Revision 1.5  1997/03/21 05:01:29  vikas
 * cleanup for release
 *
 * Revision 1.4  1994/05/16 01:27:20  vikas
 * Rewrite to be more consistent with the rest of the nocol code
 * (calls nocol_done(), consistent prognm, etc. variable names).
 *
 * Revision 1.3  1994/01/10  19:32:47  aggarwal
 * Typecast malloc() to char *
 *
 * Revision 1.1  1993/10/08  19:16:55  aggarwal
 * Initial revision
 *
 *
 *
 */
#include "nocol.h"
#include <sys/file.h>			/* for access() defines */
#include <string.h>

#ifndef SLEEPTIME
# define SLEEPTIME  15			/* after each popen()  */
#endif

char	*prognm;
int	debug;

static char  *configfile, *datafile ;	/* shld be initially NULL */
static char  *execprogpath, *execprog, *execargs ; /* program to be executed */
extern char *skip_spaces() ;		/* in libnocol */

main (ac, av)
     int ac;
     char **av;
{
    extern char *optarg;
    extern int  optind;
    int         c;

    prognm = av[0] ;

    while ((c = getopt(ac, av, "da:e:")) != EOF)
      switch(c)
      {
       case 'd':
          debug++;
          break;
       case 'a':
          execargs = optarg ;		/* command line args to execprog */
          break ;
       case 'e':
	  execprogpath = optarg ;	/* full path to prog being run */
          break ;
       case '?':
       default:
          fprintf(stderr, "%s: Unknown flag: %c\n", prognm, optarg);
          fprintf(stderr, "Usage: %s [-d] [-e <prog to exec>] ", prognm);
          fprintf(stderr, "[-a <'args to above prog'>]\n");
	  fprintf(stderr, "OR: use as filter in a pipe process flow\n");
          exit (1);
      }

    switch ( ac - optind )
    {
     case 0:                                    /* default input file */
        break ;
     default:
        fprintf (stderr, "%s: Bad arg '%s'\nUse -h for help, exiting\n",
		 prognm, av[optind]);
    }

    if (debug)
#ifndef DEBUG
      fprintf(stderr,"%s: WARNING- program NOT compiled with DEBUG option\n",
	      prognm);
#else
    fprintf(stderr, "%s: Debug turned on\n", prognm);
#endif

    if (debug)
      fprintf(stderr, "(debug) %s:  execprogpath= '%s', execargs= '%s'\n\n",
	      prognm, execprogpath, execargs);

    if (execprogpath != NULL && *execprogpath != '\0')
      mastermode();
    else			/* read from stdin  */
      slavemode() ;

}	/* end:  main()  */

/*+ 
 * FUNCTION:
 * 	If this program is going to run the program specified on the
 * command line (execprogpath). Calls nocol_startup() and daemonizes.
 */
mastermode()
{
    int  fd ;	    	/* data file descriptor */
    char cmd[1024];	/* command to be run */
    FILE *cmdf ;	/* for popen() */

    if (access(execprogpath, F_OK | X_OK) != 0) /* check if prog executable  */
    {
	fprintf(stderr, "%s: Fatal error in access() ", prognm) ;
        perror (execprogpath);
        exit(1);
    }

    if ((execprog = (char *)strrchr (execprogpath, '/')) == NULL)
      execprog = execprogpath ;		/* no path in program name */
    else
      execprog++ ;                                /* skip leading '/' */

    prognm = execprog ;			/* use exec program's name */
    nocol_startup(&configfile, &datafile);

    if (!debug)
      daemonize() ;

    while (1)
    {
      if ((fd = open(datafile, O_WRONLY | O_CREAT, DATAFILE_MODE)) < 0)
      {
	perror(datafile);
	fatal ("Open failed, exiting...");
      }

      sprintf(cmd, "%s %s\0", execprogpath, execargs);
      if ((cmdf = popen (cmd, "r")) == NULL)
      {
	perror("popen");
	fatal("Exiting...");
      }

      dostream(cmdf, fd);

      if (fd >= 0) close (fd);
      if (cmdf)    pclose(cmdf);

#ifdef DEBUG
      fprintf(stderr, "(debug) mastermode: sleeping\n");
#endif
      sleep (SLEEPTIME);
    }	/* end while(1) */

}	/* end:  mastermode() */

/*+ 
 * FUNCTION:
 * 	When this program is called at the end of a pipe, then it has
 * to read nocol event text lines from the stdin.
 * Does not 'sleep' or anything because the invoking process will
 * probably want control right away.
 */

slavemode()
{
#ifdef DEBUG
    if (debug)
      fprintf(stderr, "(debug) %s: Entered slavemode, reading from stdin\n",
	      prognm);
#endif

    dostream(stdin,fileno(stdout));	/* work with stdin/stdout */

}	/* end:  slavemode()  */

/*+ 
 * FUNCTION:
 * 	Read text lines from the indicated file stream and write out
 * to the open file descriptor  in the nocol format
*/
dostream(fptr, fd)
     FILE *fptr ;		/* to read text lines from */
     int  fd;			/* to write nocol structures to */
{
    EVENT *pv ;
    char  line[1024] ;		/* read line */

    if (fptr == NULL  ||  fd < 0)
      return(0);


    while (fgets(line, sizeof (line), fptr) != NULL)
    {
	int   doneappend =0, dolog =0;
	char  *tstr ;	/* temp pointer */

	if ((tstr = strrchr(line, '\n')) )
	  *tstr = '\0' ;	/* chop off newline */
#ifdef DEBUG
	if (debug)
	  fprintf(stderr, "(debug) dostream: read '%s'\n", line);
#endif
	tstr = line ;
	tstr = skip_spaces(tstr);

	/*
	 * look for keywords such as APPEND or LOG
	 */
	if (strncasecmp(tstr, "APPEND", strlen("APPEND")) == 0)
	{
	    if (!doneappend)
	    {
		/* Check if output device can be lseeked, etc. ?? */
		if (debug)
		  fprintf(stderr, "(debug) dostream: scanned APPEND, appending to output\n") ;
		lseek(fd, 0L, SEEK_END);	/* end of file */
		doneappend = 1;
	    }
	    tstr = tstr + strlen("APPEND") ;
	    tstr = skip_spaces(tstr);
	}

	if (strncasecmp(tstr, "LOG", strlen("LOG")) == 0)
	{
	    dolog = 1;
	    tstr = tstr + strlen("LOG") ;
	    tstr = (char *)skip_spaces(tstr);
	}

	pv= (EVENT *)logstr_to_event(tstr) ;	/* create an event struct */
	if (pv == NULL)
	  return (-1);

	if (dolog)
	  eventlog(pv);

	if (write(fd, pv, sizeof (EVENT)) != sizeof (EVENT) )
	{
	    perror("write");
	    return(-1);
	}

    }	/* end:  while()  */

    
}  /* end:  dostream()  */


/*
 * Become a daemon: go into the background, reset our process group,
 * and clobber the current directory and stdin, stdout, stderr.
 * Ideas for what needs to be done more or less stolen from BSD sources.
 * Exits on failure, and complains vigorously!
 */
daemonize()
{
    int  retval, fd, errfd;
    
    if ((retval=fork()) < 0)                    /* Go into the background */
      fatal("daemon(): fork() failed");
    else if (retval)
      exit(0);					/* parent process */
    
    if ((int) setsid() < 0)                     /* Reset the process group */
      fatal("daemon(): setsid() failed");
    
    if (chdir("/") < 0)                         /* Clobber current directory */
      fatal("daemon(): chdir() failed");
    
    fd = open("/dev/null", O_RDWR, 0);          /* Clobber stdin, stdout, .. */

    if (fd < 0)
      fatal("daemon(): open() failed for errorfile");
    if (dup2(fd, 1) < 0 || dup2(fd, 2) < 0)	/* dup stdout and stderr */
      fatal("daemon(): dup2() failed");

    return (1);			/* never reached */
}	/* daemon() */
    

fatal(s)	
char *s;
{
    fprintf(stderr, "(fatal) %s: ", prognm) ;
    if (s)
      fprintf(stderr, "%s\n", s);
    else
      fprintf(stderr, "\n");

    nocol_done();	/* does an exit for us */

}		/* fatal()  */

