/*
 * $Header: /home/vikas/src/nocol/nsmon/RCS/nsmon.c,v 1.7 1994/05/16 02:04:32 vikas Exp $
 */
/*+ 
** FUNCTION:
** 	Make a domain nameserver query and send it off to the server.
** Used as part of the 'nocol' nsmon nameserver monitoring program.
** Query options customized to test if the nameserver is up and running.
**
**
** AUTHOR
**	S. Spencer Sun, Princeton Univ. / JvNCnet, June 1992
**	Modifications & cleanup:	vikas@navya.com, Jan 94
**/

/* Copyright 1992 JvNCnet */

/*
 * $Log: nsmon.c,v $
 * Revision 1.7  1994/05/16 02:04:32  vikas
 * Deleted all the includes since they are in nocol.h and nsmon.h
 *
 * Revision 1.6  1994/01/10  19:47:33  aggarwal
 * More organized. Was ignoring the return codes from the res_query, hence
 * would get no answer NO_NSDATA. More explanations added. More debugging.
 *
 * Revision 1.5  1993/11/03  20:51:03  aggarwal
 * Added ifdef for h_addr (defined in netdb.h) in case its defined.
 *
 * Revision 1.3  1993/10/29  23:15:25  aggarwal
 * Changed to use UDP instead of TCP. The timeout mechanism in
 * TCP was kinda ugly (using setjmp)
 *
 * Revision 1.1  1992/06/11  05:03:14  aggarwal
 * Initial revision
 *
 */

#include "nsmon.h"

/*
 * nsmon -- builds the query using res_mkquery() and then sends it using
 * res_send().
 *
 * input parameters:
 *      server: address of the nameserver to be queried.
 *      request: query to be passed to nameserver.  If type is T_SOA then
 *              this needs to be a domain name
 *      class: a class, probably C_IN, as defined in <arpa/nameser.h>
 *      type: query type, probably T_SOA, as defined in <arpa/nameser.h>
 *      timeout: if no response in this many seconds, give up
 *	aa_only: if requesting an Authoritative answer only
 *	debug:   if debug mode is to be enabled
 *
 * return values:
 *	integer (defined in nsmon.h)
 */

nsmon (server, request, class, type, timeout, aa_only, debug)
  char *server, *request;
  int class, type, timeout, aa_only, debug;
{
  int n, ch;
  HEADER *hp;				/* struct defined in arpa/nameser.h */
  char buf[BUFSIZ], answer[BUFSIZ], *p;
  struct sockaddr_in sa, *psin ;
/*  extern struct state _res;		/* defined in resolv.h */
  
  psin = &(_res.nsaddr) ;
  /* insert server address into _res structure */
  bzero(psin, sizeof (*psin));
  psin->sin_family = AF_INET ;
  psin->sin_port = htons(NAMESERVER_PORT);
  if (isdigit (*server))
    psin->sin_addr.s_addr = inet_addr(server);
  else
  {
      struct hostent *thp ;	/* gethostbyname returns in_addr structures */
      if ((thp = gethostbyname(server)) == NULL)
      {
	  fprintf (stderr, "gethostbyname () failed for '%s'- ", server);
	  perror("nsmon");
	  return NS_ERROR ;
      }
#ifdef h_addr		/* in netdb.h */
      bcopy ((char *)thp->h_addr, (char *)&(psin->sin_addr), thp->h_length) ;
#else
      bcopy ((char *)thp->h_addr_list[0], (char *)&(psin->sin_addr), 
	     thp->h_length) ;
#endif
  }

  if (debug) 
    fprintf (stderr, "(debug) Nameserver is %s, port %d\n",
      inet_ntoa(_res.nsaddr.sin_addr), ntohs(_res.nsaddr.sin_port));

  /*
   * set various flags for _res options. We don't call res_init() to get
   * finer control over the options.
   */
  _res.options = RES_INIT;		/* don't call res_init() */
/*  _res.options |= RES_USEVC ;		/* use TCP and not UDP */
  _res.options |= RES_PRIMARY;		/* only query primary server */
  _res.options &= ~RES_RECURSE ;	/* no recursion */
  _res.options &= ~RES_DNSRCH ;		/* don't search up domain tree */
  if (aa_only)
    _res.options |= RES_AAONLY; 	/* only authoritative answers */
  if (timeout)
     _res.retrans = timeout ;		/* timeout in seconds */

  _res.nscount = 1 ;			/* number of nameservers */
  _res.retry = 2; 

  if (debug > 1) {
    _res.options |= RES_DEBUG;
    fprintf(stderr, "(debug):     _res.options = %ld\n", _res.options);
    fprintf(stderr, "(debug):     _res.defdname = %s\n", _res.defdname);
    fprintf(stderr, "(debug): About to call res_mkquery()\n");
    fflush(stderr);
  }

  /*
   * First ask res_mkquery() to build a query structure. 'QUERY' is an
   * opcode defined in nameser.h.
   */
  n = res_mkquery(/* opcode */QUERY, request, class, type, 
		  (char *)NULL, 0, NULL, (char *)buf, sizeof(buf));
  /* query in buf */
  
  if (n < 0) {
      if (debug) {
	  fprintf(stderr, "(debug) nsmon: problem with site %s\n", server);
	  perror("res_mkquery");
      }
      return NS_ERROR;
  }

  if (debug > 1) {
      fprintf(stderr, "(debug):     _res.options = %ld\n", _res.options);
      fprintf(stderr, "(debug):     _res.defdname = %s\n", _res.defdname);
      fprintf(stderr, "(debug): About to call res_send()\n");
      fflush(stderr);
  }

  /* Now send the query using res_send() */

  n = res_send(buf, n, answer, sizeof(answer) ) ;
  
  if (n < 0) {
      if (debug) {
	  printf("(debug) nsmon: problem with site %s\n",  server);
	  perror("res_send");
      }
      return NS_ERROR;
  }
  else
    if (debug > 1)
      fprintf (stderr, "(debug) nsmon: Returned from res_send\n");
  
  hp = (HEADER *)answer ;
  
  if (hp->rcode != NOERROR) {
      if (debug)
	fprintf(stderr, "(debug): got rcode of %s from nameserver\n",
		Rcode_Bindings[hp->rcode]);
      return NS_ERROR;
  }

  if (ntohs(hp->ancount) == 0)	/* no error, but no data either */
  {
      if (debug)
	fprintf(stderr, "(debug): got rcode NOERROR, but zero answers (no data)\n");
      return NO_NSDATA;
  }

  /*
   * The AA_ONLY option is not implemented by most resolvers, so we have
   * to check if the answer is actually what we asked for.
   */
  if (aa_only && !(hp->aa))
  {
      if (debug)
	fprintf(stderr, "(debug): requested Authoritative, got non_aa data\n");
      return NOT_AUTHORITATIVE;
  }

  
  return ALL_OK;		/* Don't check the response, must be okay */

}			/* end nsmon() */
