/* $Header: /home/vikas/src/nocol/portmon/RCS/portmon.c,v 1.8 1998/08/13 11:31:48 vikas Exp $  */

/* Copyright 1994 Vikas Aggarwal, vikas@navya.com */

/*+ 		portmon
 * FUNCTION:
 * 	Monitor TCP ports and reponses for NOCOL. It can send a string
 * and then parse the responses against a list. Each response can be
 * assigned a severity. Checks simple port connectivity if given a NULL
 * response string.
 *
 * CAVEATS:
 *	1) Uses 'strstr' and not a real regular expression. Case sensitive
 *	2) Looks only at the first buffer of the response unless using the
 *	   timeouts to calculate the response time.
 *	3) Does not implement milli-second timers while reading responses
 *	   since that level of accuracy is not really required.
 *
 *  	Vikas Aggarwal, -vikas@navya.com
 */


/*
 * $Log: portmon.c,v $
 * Revision 1.8  1998/08/13 11:31:48  vikas
 * Changed goto into a while() loop. Was core dumping on Solaris with
 * gcc v2.8.1. Patch sent in by durrell@innocence.com.
 *
 * Revision 1.7  1998/07/31 18:32:00  vikas
 * bug in timeout array and increased size of read buffer
 *
 * Revision 1.6  1997/10/19 03:17:17  vikas
 * - cleanup for releasing with nocol v4.2
 * - added code for checking response times
 *
 * Revision 1.5  1994/05/16  02:12:55  vikas
 * bugfix from David Meyer (meyer@phloem.uoregon.edu) to distinguish
 * between the alarm going off and read error (n=0 vs. n<0)
 *
 * Revision 1.4  1994/05/09  01:03:37  vikas
 * Rewrite to use new nocol library 'startup_nocol' function.
 *
 * Revision 1.3  1994/01/10  20:52:20  aggarwal
 * Added typcast for malloc. Changed fgetline() to fgetLine()
 * Fixed strdup to handle NULL strings. Added comments if just checking
 * connectivity.
 *
 * Revision 1.2  1993/11/03  20:51:03  aggarwal
 * Added ifdef for h_addr (defined in netdb.h) in case its defined.
 *
 * Revision 1.1  1993/10/30  03:39:04  aggarwal
 * Initial revision
 *
 */

/*  */
#ifndef lint
static char rcsid[] = "$Id: portmon.c,v 1.8 1998/08/13 11:31:48 vikas Exp $" ;
#endif

#include "portmon.h"

char	*prognm;
int	debug;

static int 	maxseverity, fdout, pollinterval ;
static int 	timedout, sock ;
static int	rtimeout = RTIMEOUT;		/* read timeout in seconds */
static char	*configfile, *datafile;
static char 	*sender;

extern char	*skip_spaces() ;		/* in libnocol */

main(ac, av)
  int ac;
  char **av;
{
  extern char *optarg;
  extern int  optind;
  int         c ;

  if ((prognm = (char *)strrchr (av[0], '/')) == NULL)
    prognm = av[0] ;                        /* no path in program name */
  else
    prognm++ ;                                /* skip leading '/' */

#ifdef SENDER
  sender = SENDER ;
#else                                           /* delete the directory name */
  sender = prognm ;                         /* no path in program name */
#endif


  pollinterval = POLLINTERVAL ;

  while ((c = getopt(ac, av, "df:")) != EOF)
    switch(c)
    {
    case 'd':
      debug++;
      break;
    case 'f':
      configfile = optarg ;
      break ;
    case '?':
    default:
      fprintf(stderr, "%s: Unknown flag: %c\n", prognm, optarg);
      fprintf(stderr, "Usage: %s [-d] [-f <config file>\n] ", prognm);
      exit (1);
    }

  nocol_startup(&configfile, &datafile);

  if ( (fdout = open(datafile, O_RDWR|O_CREAT|O_TRUNC, DATAFILE_MODE)) < 0)
  {
    fprintf(stderr, "(%s) ERROR in open datafile ", prognm);
    perror (datafile);
    nocol_done();
  }

  if (readconfig(fdout) == -1)
    nocol_done();

  lseek(fdout, (off_t)0, SEEK_SET);	/* rewind file */
  while (1)	/* forever */
  {
    for (c = 0; hostarray[c].hname; ++c)
    {
      int status ;
      EVENT v;

      read(fdout, &v, sizeof v);
      status = checkports(&(hostarray[c])) ;
      if (status == -1)
	fprintf (stderr, "%s: Error in checkports, skipping site %s\n",
		 prognm, hostarray[c].hname);
      else
      {
	update_event(&v, status, /* value */ (u_long)status, maxseverity) ;
	lseek(fdout, -(off_t)sizeof(v), SEEK_CUR);
	write(fdout, (char *)&v, sizeof(v));
      }
    }	/* for () */

    lseek(fdout, (off_t)0, SEEK_SET);	/* rewind file */

    if (debug)
      fprintf(stderr, "(debug) %s: sleeping for %d...zzz\n", 
	      prognm,pollinterval);
#ifdef SVR4
    bsdsignal(SIGALRM, SIG_DFL);
#else
    signal(SIGALRM, SIG_DFL);		/* wake-up, in case blocked */
#endif
    sleep (pollinterval);

  }	/* while (forever) */

}	/* end:  main()  */


/*+ 
 * FUNCTION:
 * 	Checks the port for the structure _harray passed to it. Sets
 * global maxseverity. Return value is the 'status' to be used in
 * update_event().
 */
checkports(h)
  struct _harray  *h;
{
  int status, n;
  long elapsed_time = 0;
  char readbuf[BUFSIZ * 2];
  struct sockaddr_in sin;

  if (debug)
    fprintf (stderr, "(debug) %s: Checking site '%s:%d'\n",
	     prognm, h->hname, h->port);

  if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0)
  {
    perror("socket");
    return (-1);
  }

  bzero(&sin, sizeof(sin));
  sin.sin_family = AF_INET;
  sin.sin_port = htons(h->port);	/* port number to use */

  if (isdigit(*(h->ipaddr)) ) 	/* given an address, not a name */
    sin.sin_addr.s_addr = inet_addr(h->ipaddr);
  else
  {
    struct hostent  *hp ;
    if ((hp = gethostbyname(h->ipaddr)) == NULL)
    {
      perror(prognm);
      fprintf(stderr, "gethostbyname() failed for %s\n", h->ipaddr);
      return(-1);
    }
#ifdef h_addr	/* in netdb.h */
    sin.sin_addr.s_addr = inet_addr(hp->h_addr) ;
#else
    sin.sin_addr.s_addr = inet_addr(hp->h_addr_list[0]) ;
#endif
  }

  if (sin.sin_addr.s_addr <= 0)		/* error */
  {
    perror("inet_addr() failed");
    return(-1);				/* fatal error */
  }

  /* this will try until the default TCP timeout */
  if (connect(sock, (struct sockaddr *)&sin, sizeof(sin)) < 0)
  {
    if (debug)
      perror("client: AF_INET connect() failed");
    close(sock);
    maxseverity = h->maxseverity;
    return(0);
  }

  if (debug)
    fprintf(stderr, "(debug) Connected: '%s:%d'\n", h->hname, h->port);
    
  /*
   *	- If send-string, read and discard any initial connect string
   *	  then send off the send-string
   *	- If no recv-strings, then return OK
   *	- else Read from socket, and compare with the list of recv-strings
   */

  if (h->send && *(h->send))
  {				/* if something to send */
    if (debug)
      fprintf(stderr, "(debug) Looking for any initial connect string\n");
    if ((n = timeout_read (readbuf, sizeof readbuf, CONNECT_TIMEOUT)) < 0)
    {
      if (debug && !timedout)	/* some wierd read error, ignore here */
	fprintf(stderr, "(debug) %s: Ignoring read() error\n", prognm);
    }
    else if (debug)
      fprintf(stderr, " (debug) %s: ignoring connect string '%s'\n",
	      prognm, readbuf );

    if (write(sock, h->send, strlen(h->send)) != strlen(h->send))
    {
      if (debug)
      {
	fprintf(stderr, 
		"Send string error for '%s:%d'\n", h->hname, h->port);
	perror("write");
      }
      close (sock);
      maxseverity = h->maxseverity;
      return(0);	/* mark as down */
    }	/* end if (couldn't write)  */

    if (debug)
      fprintf(stderr, " (debug) %s: host %s- sent string '%s'\n",
	      prognm, h->hname, h->send) ;

  }	/* end:  if (something to send) */


  if (h->responses[0].response == NULL)	/* no responses to check */
  {
    maxseverity = E_INFO;
    close (sock);
    return(1);
  }

  for ( ; ; )
  {			/* forever */
    int rto;
    time_t st_time = time((time_t *)0);

    rto =  (h->timeouts[2] ? (h->timeouts[2] - elapsed_time) : rtimeout);
    if (rto <= 0 || timeout_read(readbuf, sizeof readbuf, rto) <= 0)
    {
      if (debug)
      {
	fprintf(stderr, "read error for '%s:%d'", h->hname, h->port);
	if (timedout)	  fprintf(stderr, "- timeout\n");
	else		  perror("read");
      }
      maxseverity = h->maxseverity;
      elapsed_time = 9999;		/* artifically high number */
      break;
    }
    else
    {
      elapsed_time += (time((time_t *)0) - st_time);	/* stop timer */
      maxseverity = check_resp(readbuf, h->responses);
    }

    if (maxseverity != -1)    		/* response in our list */
      break;

    if (h->timeouts[0] == 0)
    {
      maxseverity = h->maxseverity;	/* use default severity */
      break;	/* out of forever */
    }
  }	/* end forever() */

  close(sock);
  if (h->timeouts[0] != 0)		/* we are checking port speed */
  {
    if (debug) fprintf(stderr,"(debug) elapsed time= %d secs\n", elapsed_time);
    if (elapsed_time < h->timeouts[0])		maxseverity = E_INFO;
    else if (elapsed_time < h->timeouts[1])	maxseverity = E_WARNING;
    else if (elapsed_time < h->timeouts[2])	maxseverity = E_ERROR;
    else maxseverity = E_CRITICAL;
  }
  if (debug)
    fprintf(stderr, " (debug) checkport(): returning severity %d\n",
	    maxseverity) ;
  if (maxseverity == E_INFO)
    return(1);
  else
    return(0);		/* status is down */

}	/* end:  checkports()  */

/*+ 
 * FUNCTION:
 * Read from socket into buffer and put terminating NULL.
 * Wait max timeout secs for the data to come in.
 * Modifies global variable 'timedout' -
 * Returns the number of bytes read.
 * Returns -2 if timed out . Returns -1 if error occured
 */
timeout_read(buf, bufsiz, timeout)
  char *buf;
  int bufsiz, timeout ;
{
  int n, len = bufsiz ;
  char *cp;
  fd_set  fdvar;
  struct timeval tval ;

  cp = buf ; 
  timedout = 0;

  /* 
   * Use select to block for timeout secs.
   * If no data within this time set timedout to 1 and return -2.
   * Else read data
   */
  tval.tv_sec = timeout;
  tval.tv_usec = 0;
  FD_ZERO(&fdvar);
  FD_SET(sock, &fdvar);

  if ( (n = select(sock+1, &fdvar, NULL, NULL, &tval)) <= 0)
  {
    if (n < 0 && errno != EINTR)
    {
      perror("select() ");
      return (-1);
    }
    else
    {
      timedout = 1;
      if(debug)   fprintf(stderr, "timeout_read(): select() timed out\n");
      return (-2);	/* Indicates timeout */
    }
  }

  /* read a LINE from global socket (until newline) */
  if(debug)
    fprintf(stderr, "(debug) timeout_read(): Reading a line from socket..\n");

  while (len != 0 && (n = read (sock, cp, len)) > 0)
  {
    cp += (n - 1);	/* increment pointer to last byte read */
    len -= n;
    if (*cp == '\n'  || *cp == '\0')
      break ;
    else
      cp++ ;	/* increment past last byte read */
  }


  if (n <= 0)   len = n ;		/* some error or got alarm */
  else    len = bufsiz - len ;		/* Total number bytes read */

  if (debug)
    fprintf(stderr, "(debug) timeout_read: Read %d bytes from socket\n",len);

  buf[len > 0 ? len:0] = '\0';		/* terminate with NULL */
  return (len);				/* number bytes read else negative */
}	/* end:  timeout_read() */

     
/*+ 
 * FUNCTION:
 * 	Check the list of responses. Notice doing a strstr() which is NOT
 * case sensitive.
 */
check_resp(readstr, resarr)
  char *readstr;
  struct _response  *resarr ;
{
  if (debug)
    fprintf(stderr, " (debug) %s: Checking response '%s'\n", prognm,readstr);

  while (resarr->response)
  {
    if (strstr(readstr, resarr->response) != NULL)
    {
      if (debug)
	fprintf(stderr," (debug) check_resp(): Matched '%s'\n",
		resarr->response);
      return(resarr->severity);
    }
    ++resarr ;
  }

  if (debug)
    fprintf (stderr, " check_resp(): No response matched for site\n");

  return(-1);		/* No response matched given list */
}	/* check_resp()  */


/*+ 
 * FUNCTION:
 * 	Duplicate a string. Can handle a NULL ptr.
 */

static char *Strdup(s)
  char *s ;
{
  char *t ;

  if (s == NULL)
  {
    t = (char *)malloc(1);
    *t = '\0';
  }
  else
  {
    t = (char *)malloc(strlen(s) + 1);
    if (t != NULL)
      (char *)strcpy(t, s);
  }

  return (t);
}

/*+ 
 * FUNCTION:
 * 	Read the config file.
 * POLLINTERVAL ppp
 * HOST  <hostname>  <address>  <var>  <port> <failseverity>  <send string>
 * <severity>	response1 till end of this line
 * <severity>   response2 till end of this line
 *
 */
#define NEXTTOK  (char *)skip_spaces(strtok(NULL, " \t"))

readconfig(fdout)
  int fdout ;				/* output data filename */
{
  int i = 0, j=0;			/* array indexes */
  int maxseverity ;
  char *j1;				/* temp string pointers */
  FILE *cfd ;
  EVENT v;                            	/* Defined in NOCOL.H */
  char record[MAXLINE] ;
  struct tm *loctime ;
  time_t locclock ;                   	/* Don't use 'long'    */

  if ((cfd = fopen(configfile, "r")) == NULL)
  {
    fprintf(stderr, "%s error (init_sites) ", prognm) ;
    perror (configfile);
    return (-1);
  }

  /*
   * Fill in the static data stuff
   */
  bzero ((char *)hostarray, sizeof(hostarray));
  bzero (&v, sizeof(v)) ;

  locclock = time((time_t *)0);
  loctime = localtime((long *)&locclock);

  v.mon = loctime->tm_mon + 1;        v.day = loctime->tm_mday;
  v.hour = loctime->tm_hour;          v.min = loctime->tm_min;

  strncpy (v.sender, sender, sizeof(v.sender) - 1);

  strncpy (v.var.units, VARUNITS, sizeof (v.var.units) - 1);
  v.var.threshold = 0 ;

  v.nocop = SETF_UPDOUN (0, n_UNKNOWN); /* Set all to UNKNOWN   */
  v.severity = E_INFO ;

  /*
   * Now parse the config file
   */

  while(fgetLine(cfd,record,MAXLINE) > 0 ) 	/* keeps the \n */
  {
    static int skiphost;
    int port ;
    int checkspeed = 0;

    record[strlen(record) - 1] = '\0' ;		/* chop off newline */
    if (*record == '#' || *(skip_spaces(record)) == '\0')
      continue ;

    if (strncasecmp(record, "POLLINTERVAL", strlen("POLLINTERVAL")) == 0)
    {
      strtok(record, " \t");
      j1 = (char *)skip_spaces(strtok(NULL, "")) ;
      if (!j1 || *j1 == '\0' || (pollinterval = atoi(j1)) <= 0)
	fprintf(stderr, "%s: bad or missing pollinterval value\n", prognm);

      pollinterval = (pollinterval > 0 ? pollinterval : POLLINTERVAL);
      if (debug)
	fprintf (stderr, "(debug) %s: Pollinterval = %d\n", 
		 prognm, pollinterval);

      continue;
    }

    if (strncasecmp(record, "READTIMEOUT", strlen("READTIMEOUT")) == 0)
    {
      strtok(record, " \t");
      j1 = (char *)skip_spaces(strtok(NULL, "")) ;
      if (!j1 || *j1 == '\0' || (rtimeout = atoi(j1)) <= 0)
	fprintf(stderr, "%s: bad or missing READTIMEOUT value\n", prognm);

      rtimeout = (rtimeout > 0 ? rtimeout : RTIMEOUT);
      if (debug)
	fprintf (stderr, "(debug) %s: ReadTimeout = %d\n", 
		 prognm, rtimeout);

      continue;
    }

    if (strncasecmp(record, "HOST", 4) != 0)	/* Not HOST line */
    {
      if (skiphost)
	continue ;

      j1 = strtok(record," \t") ;			/* severity level */
      j1 = (char *)skip_spaces(j1) ;
      switch(tolower(*j1))
      {
      case 'c': case '1': maxseverity = E_CRITICAL;	break;
      case 'e': case '2': maxseverity = E_ERROR; 	break;
      case 'w': case '3': maxseverity = E_WARNING;	break;
      case 'i': case '4': maxseverity = E_INFO; 	break;
      case 't': maxseverity = E_CRITICAL; ++checkspeed; break;	/* TIME */
      default:  maxseverity = E_CRITICAL;		break ;
      }
      /* remember that the 'i' index has been incremented */
      hostarray[i-1].responses[j].severity = maxseverity ;
      j1 = (char *)skip_spaces(strtok(NULL, "")) ;
      if (checkspeed && hostarray[i-1].timeouts[0] == 0)
      {		 		/* read timeout thresholds */
	if (3 != sscanf(j1, "%u %u %u", &(hostarray[i-1].timeouts[0]),
			&(hostarray[i-1].timeouts[1]),
			&(hostarray[i-1].timeouts[2])))
	{
	  fprintf(stderr, "%s: need 3 time thresholds, invalid syntax\n",
		  prognm);
	  continue;
	}
	strtok(j1, " \t"); strtok(NULL, " \t"); strtok(NULL, " \t");
	j1 = (char *)skip_spaces(strtok(NULL, ""));
      }		/* end if(checkspeed) */
      if (j1 == NULL || *j1 == '\0')
      {
	fprintf(stderr, "%s: missing response line, skipped\n",prognm);
	fprintf(stderr, "For null responses, dont put any lines\n");
	continue;
      }
      hostarray[i-1].responses[j].response = (char *)Strdup(j1);
      j++ ;

      continue ;	/* next input line */
    }

    /*
     * Here if parsing a HOST line
     */
    j = 0;			/* and reset the response array index */
	
    skiphost = 0;		/* assume valid config */
    strtok(record, " \t");	/* skip HOST keyword */	
	    
    strncpy(v.site.name, NEXTTOK, sizeof(v.site.name) - 1);
    strncpy(v.site.addr, NEXTTOK, sizeof(v.site.addr) - 1);
    if (inet_addr(v.site.addr) == -1)        /* bad address */
    {
      fprintf(stderr,
	      "(%s): Error in addr '%s' for site '%s', ignoring\n",
	      prognm, v.site.addr, v.site.name);
      skiphost = 1;
      continue ;
    }
    strncpy(v.var.name, NEXTTOK, sizeof(v.var.name) - 1);
	
    if ((port = atoi(NEXTTOK)) == 0)
    {
      fprintf(stderr,
	      "(%s): Error in port for site '%s', ignoring\n",
	      prognm, v.site.name);
      skiphost = 1;
      continue ;
    }
	
    j1 = NEXTTOK;	/* severity level */
    if (j1 == NULL)
      j1 = "c";
    switch(tolower(*j1))
    {
    case 'c': maxseverity = E_CRITICAL; break;
    case 'e': maxseverity = E_ERROR; break;
    case 'w': maxseverity = E_WARNING; break;
    case 'i': maxseverity = E_INFO; break;
    default:  maxseverity = E_CRITICAL ; break ;
    }
	
    if (debug)
      fprintf (stderr, "Name: %s %s, VAR: %s, Port %d SEV: %d\n",
	       v.site.name,v.site.addr,v.var.name, 
	       port,maxseverity);
	
    /* the rest of string is SEND */
    hostarray[i].send = NULL ;		/* Initialize.. */
    j1 = (char *)skip_spaces(strtok(NULL, ""));
    if (j1 && *j1)
    {
      /* We NEED a newline at the end of the send strings */
      strcat(j1, "\n");
      raw2newline(j1);
      hostarray[i].send = (char *)Strdup(j1) ;

      if(debug)
	fprintf(stderr, "(debug)Sendstring: >>>%s<<<\n", hostarray[i].send);
    }

    hostarray[i].port = port;
    hostarray[i].maxseverity = maxseverity ;
    hostarray[i].hname = (char *)Strdup(v.site.name);
    hostarray[i].ipaddr = (char *)Strdup(v.site.addr);
    bzero (hostarray[i].responses, sizeof (hostarray[i].responses));
	
    if (write (fdout, (char *)&v, sizeof(v)) != sizeof(v))
    {
      fprintf(stderr, "%s (write): %s\n", prognm,sys_errlist[errno]);
      nocol_done() ;
    }
    ++i;			/* increment the index, in case */

  }	/* end: while */

  fclose (cfd);       		/* Not needed any more */    
  return(1);                          /* All OK  */

}  /* end:  init_sites() */



/*
 * Takes a string containing one or many sequences of "\n" and replaces
 * these two chars with a newline. The original string is modified.
 */
char *raw2newline(inbuf)
  char *inbuf;
{
  register char *p, *q;

  if(!inbuf)
    return (char *)NULL;

  p = q = inbuf;

  for (; ;)
  {			/* forever() */
    while (*p && *p != '\\')
      *q++ = *p++;

    if (*p == '\0') {
      *q = '\0';
      return(inbuf);
    }

    if (*(p+1) == 'n') {
      *q++ = '\n';
      p = (p+2);	/* skip \ and n */
    }
    else
      *q++ = *p++;
  }	/* end forever () */

}

